package org.pdb.ormapping;

import org.apache.commons.lang.builder.HashCodeBuilder;

/**
 *  Data items in the ATOM_SITE category record details about
 *  the atom sites in a macromolecular crystal structure, such as
 *  the positional coordinates, atomic displacement parameters,
 *  magnetic moments and directions.
 *  
 *  The data items for describing anisotropic atomic
 *  displacement factors are only used if the corresponding items
 *  are not given in the ATOM_SITE_ANISOTROP category.
 *  
 * <pre>
 *  Example 1 - based on PDB entry 5HVP and laboratory records for the
 *  structure corresponding to PDB entry 5HVP.
 *  <PDBx:atom_siteCategory>
 *  <PDBx:atom_site id="1">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>N</PDBx:type_symbol>
 *  <PDBx:label_atom_id>N</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>VAL</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>11</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>25.369</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>30.691</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>11.795</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>17.93</PDBx:B_iso_or_equiv>
 *  <PDBx:auth_seq_id>11</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="2">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>CA</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>VAL</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>11</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>25.970</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>31.965</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>12.332</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>17.75</PDBx:B_iso_or_equiv>
 *  <PDBx:auth_seq_id>11</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="3">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>C</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>VAL</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>11</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>25.569</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>32.010</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>13.808</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>17.83</PDBx:B_iso_or_equiv>
 *  <PDBx:auth_seq_id>11</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="4">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>O</PDBx:type_symbol>
 *  <PDBx:label_atom_id>O</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>VAL</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>11</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>24.735</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>31.190</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>14.167</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>17.53</PDBx:B_iso_or_equiv>
 *  <PDBx:auth_seq_id>11</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="5">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>CB</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>VAL</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>11</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>25.379</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>33.146</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>11.540</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>17.66</PDBx:B_iso_or_equiv>
 *  <PDBx:auth_seq_id>11</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="6">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>CG1</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>VAL</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>11</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>25.584</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>33.034</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>10.030</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>18.86</PDBx:B_iso_or_equiv>
 *  <PDBx:auth_seq_id>11</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="7">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>CG2</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>VAL</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>11</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>23.933</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>33.309</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>11.872</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>17.12</PDBx:B_iso_or_equiv>
 *  <PDBx:auth_seq_id>11</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="8">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>N</PDBx:type_symbol>
 *  <PDBx:label_atom_id>N</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>THR</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>12</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>26.095</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>32.930</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>14.590</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>18.97</PDBx:B_iso_or_equiv>
 *  <PDBx:footnote_id>4</PDBx:footnote_id>
 *  <PDBx:auth_seq_id>12</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="9">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>CA</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>THR</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>12</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>25.734</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>32.995</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>16.032</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>19.80</PDBx:B_iso_or_equiv>
 *  <PDBx:footnote_id>4</PDBx:footnote_id>
 *  <PDBx:auth_seq_id>12</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="10">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>C</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>THR</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>12</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>24.695</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>34.106</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>16.113</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>20.92</PDBx:B_iso_or_equiv>
 *  <PDBx:footnote_id>4</PDBx:footnote_id>
 *  <PDBx:auth_seq_id>12</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="11">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>O</PDBx:type_symbol>
 *  <PDBx:label_atom_id>O</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>THR</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>12</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>24.869</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>35.118</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>15.421</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>21.84</PDBx:B_iso_or_equiv>
 *  <PDBx:footnote_id>4</PDBx:footnote_id>
 *  <PDBx:auth_seq_id>12</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="12">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>CB</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>THR</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>12</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>26.911</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>33.346</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>17.018</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>20.51</PDBx:B_iso_or_equiv>
 *  <PDBx:footnote_id>4</PDBx:footnote_id>
 *  <PDBx:auth_seq_id>12</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="13">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>O</PDBx:type_symbol>
 *  <PDBx:label_atom_id>OG1</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>THR</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>12</PDBx:label_seq_id>
 *  <PDBx:label_alt_id>3</PDBx:label_alt_id>
 *  <PDBx:Cartn_x>27.946</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>33.921</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>16.183</PDBx:Cartn_z>
 *  <PDBx:occupancy>0.50</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>20.29</PDBx:B_iso_or_equiv>
 *  <PDBx:footnote_id>4</PDBx:footnote_id>
 *  <PDBx:auth_seq_id>12</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="14">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>O</PDBx:type_symbol>
 *  <PDBx:label_atom_id>OG1</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>THR</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>12</PDBx:label_seq_id>
 *  <PDBx:label_alt_id>4</PDBx:label_alt_id>
 *  <PDBx:Cartn_x>27.769</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>32.142</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>17.103</PDBx:Cartn_z>
 *  <PDBx:occupancy>0.50</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>20.59</PDBx:B_iso_or_equiv>
 *  <PDBx:footnote_id>4</PDBx:footnote_id>
 *  <PDBx:auth_seq_id>12</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="15">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>CG2</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>THR</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>12</PDBx:label_seq_id>
 *  <PDBx:label_alt_id>3</PDBx:label_alt_id>
 *  <PDBx:Cartn_x>27.418</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>32.181</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>17.878</PDBx:Cartn_z>
 *  <PDBx:occupancy>0.50</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>20.47</PDBx:B_iso_or_equiv>
 *  <PDBx:footnote_id>4</PDBx:footnote_id>
 *  <PDBx:auth_seq_id>12</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="16">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>CG2</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>THR</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>12</PDBx:label_seq_id>
 *  <PDBx:label_alt_id>4</PDBx:label_alt_id>
 *  <PDBx:Cartn_x>26.489</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>33.778</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>18.426</PDBx:Cartn_z>
 *  <PDBx:occupancy>0.50</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>20.00</PDBx:B_iso_or_equiv>
 *  <PDBx:footnote_id>4</PDBx:footnote_id>
 *  <PDBx:auth_seq_id>12</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="17">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>N</PDBx:type_symbol>
 *  <PDBx:label_atom_id>N</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>ILE</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>13</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>23.664</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>33.855</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>16.884</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>22.08</PDBx:B_iso_or_equiv>
 *  <PDBx:auth_seq_id>13</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="18">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>CA</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>ILE</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>13</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>22.623</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>34.850</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>17.093</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>23.44</PDBx:B_iso_or_equiv>
 *  <PDBx:auth_seq_id>13</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="19">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>C</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>ILE</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>13</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>22.657</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>35.113</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>18.610</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>25.77</PDBx:B_iso_or_equiv>
 *  <PDBx:auth_seq_id>13</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="20">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>O</PDBx:type_symbol>
 *  <PDBx:label_atom_id>O</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>ILE</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>13</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>23.123</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>34.250</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>19.406</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>26.28</PDBx:B_iso_or_equiv>
 *  <PDBx:auth_seq_id>13</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="21">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>CB</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>ILE</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>13</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>21.236</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>34.463</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>16.492</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>22.67</PDBx:B_iso_or_equiv>
 *  <PDBx:auth_seq_id>13</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="22">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>CG1</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>ILE</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>13</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>20.478</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>33.469</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>17.371</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>22.14</PDBx:B_iso_or_equiv>
 *  <PDBx:auth_seq_id>13</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="23">
 *  <PDBx:group_PDB>ATOM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>CG2</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>ILE</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>A</PDBx:label_asym_id>
 *  <PDBx:label_seq_id>13</PDBx:label_seq_id>
 *  <PDBx:label_alt_id xsi:nil="true" />
 *  <PDBx:Cartn_x>21.357</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>33.986</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>15.016</PDBx:Cartn_z>
 *  <PDBx:occupancy>1.00</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>21.75</PDBx:B_iso_or_equiv>
 *  <PDBx:auth_seq_id>13</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="101">
 *  <PDBx:group_PDB>HETATM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>C1</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>APS</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>C</PDBx:label_asym_id>
 *  <PDBx:label_seq_id xsi:nil="true" />
 *  <PDBx:label_alt_id>1</PDBx:label_alt_id>
 *  <PDBx:Cartn_x>4.171</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>29.012</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>7.116</PDBx:Cartn_z>
 *  <PDBx:occupancy>0.58</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>17.27</PDBx:B_iso_or_equiv>
 *  <PDBx:footnote_id>1</PDBx:footnote_id>
 *  <PDBx:auth_seq_id>300</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="102">
 *  <PDBx:group_PDB>HETATM</PDBx:group_PDB>
 *  <PDBx:type_symbol>C</PDBx:type_symbol>
 *  <PDBx:label_atom_id>C2</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>APS</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>C</PDBx:label_asym_id>
 *  <PDBx:label_seq_id xsi:nil="true" />
 *  <PDBx:label_alt_id>1</PDBx:label_alt_id>
 *  <PDBx:Cartn_x>4.949</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>27.758</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>6.793</PDBx:Cartn_z>
 *  <PDBx:occupancy>0.58</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>16.95</PDBx:B_iso_or_equiv>
 *  <PDBx:footnote_id>1</PDBx:footnote_id>
 *  <PDBx:auth_seq_id>300</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="103">
 *  <PDBx:group_PDB>HETATM</PDBx:group_PDB>
 *  <PDBx:type_symbol>O</PDBx:type_symbol>
 *  <PDBx:label_atom_id>O3</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>APS</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>C</PDBx:label_asym_id>
 *  <PDBx:label_seq_id xsi:nil="true" />
 *  <PDBx:label_alt_id>1</PDBx:label_alt_id>
 *  <PDBx:Cartn_x>4.800</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>26.678</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>7.393</PDBx:Cartn_z>
 *  <PDBx:occupancy>0.58</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>16.85</PDBx:B_iso_or_equiv>
 *  <PDBx:footnote_id>1</PDBx:footnote_id>
 *  <PDBx:auth_seq_id>300</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  <PDBx:atom_site id="104">
 *  <PDBx:group_PDB>HETATM</PDBx:group_PDB>
 *  <PDBx:type_symbol>N</PDBx:type_symbol>
 *  <PDBx:label_atom_id>N4</PDBx:label_atom_id>
 *  <PDBx:label_comp_id>APS</PDBx:label_comp_id>
 *  <PDBx:label_asym_id>C</PDBx:label_asym_id>
 *  <PDBx:label_seq_id xsi:nil="true" />
 *  <PDBx:label_alt_id>1</PDBx:label_alt_id>
 *  <PDBx:Cartn_x>5.930</PDBx:Cartn_x>
 *  <PDBx:Cartn_y>27.841</PDBx:Cartn_y>
 *  <PDBx:Cartn_z>5.869</PDBx:Cartn_z>
 *  <PDBx:occupancy>0.58</PDBx:occupancy>
 *  <PDBx:B_iso_or_equiv>16.43</PDBx:B_iso_or_equiv>
 *  <PDBx:footnote_id>1</PDBx:footnote_id>
 *  <PDBx:auth_seq_id>300</PDBx:auth_seq_id>
 *  </PDBx:atom_site>
 *  </PDBx:atom_siteCategory>
 * </pre>
 */
public class Atom_site extends org.pdb.beans.Atom_site implements java.io.Serializable, org.pdb.ormapping.util.OrMappingBean
{
	public static final String tagName = "atom_site";

	public void addParent(org.pdb.ormapping.util.OrMappingBean parent)
	{
		setDatablock((Datablock)parent);
	}

	/** Object identifier field */
	private Long obj_id;

	public Long getObj_id() 
	{
		return this.obj_id;
	}

	public void setObj_id(Long obj_id) 
	{
		this.obj_id = obj_id;
	}

	/** Default constructor */
	public Atom_site()
	{
	}

	/** Full constructor */
	public Atom_site(
		Datablock datablock,
		java.lang.Float B_equiv_geom_mean,
		java.lang.Float B_equiv_geom_mean_esd,
		java.lang.Float B_iso_or_equiv,
		java.lang.Float B_iso_or_equiv_esd,
		java.lang.Float Cartn_x,
		java.lang.Float Cartn_x_esd,
		java.lang.Float Cartn_y,
		java.lang.Float Cartn_y_esd,
		java.lang.Float Cartn_z,
		java.lang.Float Cartn_z_esd,
		java.lang.Float U_equiv_geom_mean,
		java.lang.Float U_equiv_geom_mean_esd,
		java.lang.Float U_iso_or_equiv,
		java.lang.Float U_iso_or_equiv_esd,
		java.lang.String Wyckoff_symbol,
		java.lang.String adp_type,
		java.lang.Float aniso_B11,
		java.lang.Float aniso_B11_esd,
		java.lang.Float aniso_B12,
		java.lang.Float aniso_B12_esd,
		java.lang.Float aniso_B13,
		java.lang.Float aniso_B13_esd,
		java.lang.Float aniso_B22,
		java.lang.Float aniso_B22_esd,
		java.lang.Float aniso_B23,
		java.lang.Float aniso_B23_esd,
		java.lang.Float aniso_B33,
		java.lang.Float aniso_B33_esd,
		java.lang.Float aniso_U11,
		java.lang.Float aniso_U11_esd,
		java.lang.Float aniso_U12,
		java.lang.Float aniso_U12_esd,
		java.lang.Float aniso_U13,
		java.lang.Float aniso_U13_esd,
		java.lang.Float aniso_U22,
		java.lang.Float aniso_U22_esd,
		java.lang.Float aniso_U23,
		java.lang.Float aniso_U23_esd,
		java.lang.Float aniso_U33,
		java.lang.Float aniso_U33_esd,
		java.lang.Float aniso_ratio,
		java.lang.Integer attached_hydrogens,
		java.lang.String auth_asym_id,
		java.lang.String auth_atom_id,
		java.lang.String auth_comp_id,
		java.lang.String auth_seq_id,
		java.lang.String calc_attached_atom,
		java.lang.String calc_flag,
		java.lang.Integer chemical_conn_number,
		java.lang.String constraints,
		java.lang.String details,
		java.lang.String disorder_assembly,
		java.lang.String disorder_group,
		java.lang.String footnote_id,
		java.lang.Float fract_x,
		java.lang.Float fract_x_esd,
		java.lang.Float fract_y,
		java.lang.Float fract_y_esd,
		java.lang.Float fract_z,
		java.lang.Float fract_z_esd,
		java.lang.String group_PDB,
		java.lang.String id_,
		java.lang.String label_alt_id,
		java.lang.String label_asym_id,
		java.lang.String label_atom_id,
		java.lang.String label_comp_id,
		java.lang.String label_entity_id,
		java.lang.Integer label_seq_id,
		java.lang.Float occupancy,
		java.lang.Float occupancy_esd,
		java.lang.String pdbx_PDB_atom_name,
		java.lang.String pdbx_PDB_ins_code,
		java.lang.Integer pdbx_PDB_model_num,
		java.lang.String pdbx_PDB_residue_name,
		java.lang.String pdbx_PDB_residue_no,
		java.lang.String pdbx_PDB_strand_id,
		java.lang.String pdbx_auth_alt_id,
		java.lang.String pdbx_auth_atom_name,
		java.lang.String refinement_flags,
		java.lang.String refinement_flags_adp,
		java.lang.String refinement_flags_occupancy,
		java.lang.String refinement_flags_posn,
		java.lang.String restraints,
		java.lang.Integer symmetry_multiplicity,
		java.lang.String thermal_displace_type,
		java.lang.String type_symbol)
	{
		this.datablock = datablock;
		this.B_equiv_geom_mean = B_equiv_geom_mean;
		this.B_equiv_geom_mean_esd = B_equiv_geom_mean_esd;
		this.B_iso_or_equiv = B_iso_or_equiv;
		this.B_iso_or_equiv_esd = B_iso_or_equiv_esd;
		this.Cartn_x = Cartn_x;
		this.Cartn_x_esd = Cartn_x_esd;
		this.Cartn_y = Cartn_y;
		this.Cartn_y_esd = Cartn_y_esd;
		this.Cartn_z = Cartn_z;
		this.Cartn_z_esd = Cartn_z_esd;
		this.U_equiv_geom_mean = U_equiv_geom_mean;
		this.U_equiv_geom_mean_esd = U_equiv_geom_mean_esd;
		this.U_iso_or_equiv = U_iso_or_equiv;
		this.U_iso_or_equiv_esd = U_iso_or_equiv_esd;
		this.Wyckoff_symbol = Wyckoff_symbol;
		this.adp_type = adp_type;
		this.aniso_B11 = aniso_B11;
		this.aniso_B11_esd = aniso_B11_esd;
		this.aniso_B12 = aniso_B12;
		this.aniso_B12_esd = aniso_B12_esd;
		this.aniso_B13 = aniso_B13;
		this.aniso_B13_esd = aniso_B13_esd;
		this.aniso_B22 = aniso_B22;
		this.aniso_B22_esd = aniso_B22_esd;
		this.aniso_B23 = aniso_B23;
		this.aniso_B23_esd = aniso_B23_esd;
		this.aniso_B33 = aniso_B33;
		this.aniso_B33_esd = aniso_B33_esd;
		this.aniso_U11 = aniso_U11;
		this.aniso_U11_esd = aniso_U11_esd;
		this.aniso_U12 = aniso_U12;
		this.aniso_U12_esd = aniso_U12_esd;
		this.aniso_U13 = aniso_U13;
		this.aniso_U13_esd = aniso_U13_esd;
		this.aniso_U22 = aniso_U22;
		this.aniso_U22_esd = aniso_U22_esd;
		this.aniso_U23 = aniso_U23;
		this.aniso_U23_esd = aniso_U23_esd;
		this.aniso_U33 = aniso_U33;
		this.aniso_U33_esd = aniso_U33_esd;
		this.aniso_ratio = aniso_ratio;
		this.attached_hydrogens = attached_hydrogens;
		this.auth_asym_id = auth_asym_id;
		this.auth_atom_id = auth_atom_id;
		this.auth_comp_id = auth_comp_id;
		this.auth_seq_id = auth_seq_id;
		this.calc_attached_atom = calc_attached_atom;
		this.calc_flag = calc_flag;
		this.chemical_conn_number = chemical_conn_number;
		this.constraints = constraints;
		this.details = details;
		this.disorder_assembly = disorder_assembly;
		this.disorder_group = disorder_group;
		this.footnote_id = footnote_id;
		this.fract_x = fract_x;
		this.fract_x_esd = fract_x_esd;
		this.fract_y = fract_y;
		this.fract_y_esd = fract_y_esd;
		this.fract_z = fract_z;
		this.fract_z_esd = fract_z_esd;
		this.group_PDB = group_PDB;
		this.id_ = id_;
		this.label_alt_id = label_alt_id;
		this.label_asym_id = label_asym_id;
		this.label_atom_id = label_atom_id;
		this.label_comp_id = label_comp_id;
		this.label_entity_id = label_entity_id;
		this.label_seq_id = label_seq_id;
		this.occupancy = occupancy;
		this.occupancy_esd = occupancy_esd;
		this.pdbx_PDB_atom_name = pdbx_PDB_atom_name;
		this.pdbx_PDB_ins_code = pdbx_PDB_ins_code;
		this.pdbx_PDB_model_num = pdbx_PDB_model_num;
		this.pdbx_PDB_residue_name = pdbx_PDB_residue_name;
		this.pdbx_PDB_residue_no = pdbx_PDB_residue_no;
		this.pdbx_PDB_strand_id = pdbx_PDB_strand_id;
		this.pdbx_auth_alt_id = pdbx_auth_alt_id;
		this.pdbx_auth_atom_name = pdbx_auth_atom_name;
		this.refinement_flags = refinement_flags;
		this.refinement_flags_adp = refinement_flags_adp;
		this.refinement_flags_occupancy = refinement_flags_occupancy;
		this.refinement_flags_posn = refinement_flags_posn;
		this.restraints = restraints;
		this.symmetry_multiplicity = symmetry_multiplicity;
		this.thermal_displace_type = thermal_displace_type;
		this.type_symbol = type_symbol;
	}

	protected Datablock datablock;


	public Datablock getDatablock() 
	{
		return datablock;
	}
 
	public void setDatablock(Datablock datablock)
	{
		this.datablock = datablock;
	}

 

	public String toString() 
	{
		StringBuffer sb = new StringBuffer();
		sb.append("Atom_site");
		sb.append("\n");
		if (B_equiv_geom_mean != null)
		{
			sb.append("B_equiv_geom_mean = ");
			sb.append(B_equiv_geom_mean);
			sb.append('\n');
		}
		if (B_equiv_geom_mean_esd != null)
		{
			sb.append("B_equiv_geom_mean_esd = ");
			sb.append(B_equiv_geom_mean_esd);
			sb.append('\n');
		}
		if (B_iso_or_equiv != null)
		{
			sb.append("B_iso_or_equiv = ");
			sb.append(B_iso_or_equiv);
			sb.append('\n');
		}
		if (B_iso_or_equiv_esd != null)
		{
			sb.append("B_iso_or_equiv_esd = ");
			sb.append(B_iso_or_equiv_esd);
			sb.append('\n');
		}
		if (Cartn_x != null)
		{
			sb.append("Cartn_x = ");
			sb.append(Cartn_x);
			sb.append('\n');
		}
		if (Cartn_x_esd != null)
		{
			sb.append("Cartn_x_esd = ");
			sb.append(Cartn_x_esd);
			sb.append('\n');
		}
		if (Cartn_y != null)
		{
			sb.append("Cartn_y = ");
			sb.append(Cartn_y);
			sb.append('\n');
		}
		if (Cartn_y_esd != null)
		{
			sb.append("Cartn_y_esd = ");
			sb.append(Cartn_y_esd);
			sb.append('\n');
		}
		if (Cartn_z != null)
		{
			sb.append("Cartn_z = ");
			sb.append(Cartn_z);
			sb.append('\n');
		}
		if (Cartn_z_esd != null)
		{
			sb.append("Cartn_z_esd = ");
			sb.append(Cartn_z_esd);
			sb.append('\n');
		}
		if (U_equiv_geom_mean != null)
		{
			sb.append("U_equiv_geom_mean = ");
			sb.append(U_equiv_geom_mean);
			sb.append('\n');
		}
		if (U_equiv_geom_mean_esd != null)
		{
			sb.append("U_equiv_geom_mean_esd = ");
			sb.append(U_equiv_geom_mean_esd);
			sb.append('\n');
		}
		if (U_iso_or_equiv != null)
		{
			sb.append("U_iso_or_equiv = ");
			sb.append(U_iso_or_equiv);
			sb.append('\n');
		}
		if (U_iso_or_equiv_esd != null)
		{
			sb.append("U_iso_or_equiv_esd = ");
			sb.append(U_iso_or_equiv_esd);
			sb.append('\n');
		}
		if (Wyckoff_symbol != null)
		{
			sb.append("Wyckoff_symbol = ");
			sb.append(Wyckoff_symbol);
			sb.append('\n');
		}
		if (adp_type != null)
		{
			sb.append("adp_type = ");
			sb.append(adp_type);
			sb.append('\n');
		}
		if (aniso_B11 != null)
		{
			sb.append("aniso_B11 = ");
			sb.append(aniso_B11);
			sb.append('\n');
		}
		if (aniso_B11_esd != null)
		{
			sb.append("aniso_B11_esd = ");
			sb.append(aniso_B11_esd);
			sb.append('\n');
		}
		if (aniso_B12 != null)
		{
			sb.append("aniso_B12 = ");
			sb.append(aniso_B12);
			sb.append('\n');
		}
		if (aniso_B12_esd != null)
		{
			sb.append("aniso_B12_esd = ");
			sb.append(aniso_B12_esd);
			sb.append('\n');
		}
		if (aniso_B13 != null)
		{
			sb.append("aniso_B13 = ");
			sb.append(aniso_B13);
			sb.append('\n');
		}
		if (aniso_B13_esd != null)
		{
			sb.append("aniso_B13_esd = ");
			sb.append(aniso_B13_esd);
			sb.append('\n');
		}
		if (aniso_B22 != null)
		{
			sb.append("aniso_B22 = ");
			sb.append(aniso_B22);
			sb.append('\n');
		}
		if (aniso_B22_esd != null)
		{
			sb.append("aniso_B22_esd = ");
			sb.append(aniso_B22_esd);
			sb.append('\n');
		}
		if (aniso_B23 != null)
		{
			sb.append("aniso_B23 = ");
			sb.append(aniso_B23);
			sb.append('\n');
		}
		if (aniso_B23_esd != null)
		{
			sb.append("aniso_B23_esd = ");
			sb.append(aniso_B23_esd);
			sb.append('\n');
		}
		if (aniso_B33 != null)
		{
			sb.append("aniso_B33 = ");
			sb.append(aniso_B33);
			sb.append('\n');
		}
		if (aniso_B33_esd != null)
		{
			sb.append("aniso_B33_esd = ");
			sb.append(aniso_B33_esd);
			sb.append('\n');
		}
		if (aniso_U11 != null)
		{
			sb.append("aniso_U11 = ");
			sb.append(aniso_U11);
			sb.append('\n');
		}
		if (aniso_U11_esd != null)
		{
			sb.append("aniso_U11_esd = ");
			sb.append(aniso_U11_esd);
			sb.append('\n');
		}
		if (aniso_U12 != null)
		{
			sb.append("aniso_U12 = ");
			sb.append(aniso_U12);
			sb.append('\n');
		}
		if (aniso_U12_esd != null)
		{
			sb.append("aniso_U12_esd = ");
			sb.append(aniso_U12_esd);
			sb.append('\n');
		}
		if (aniso_U13 != null)
		{
			sb.append("aniso_U13 = ");
			sb.append(aniso_U13);
			sb.append('\n');
		}
		if (aniso_U13_esd != null)
		{
			sb.append("aniso_U13_esd = ");
			sb.append(aniso_U13_esd);
			sb.append('\n');
		}
		if (aniso_U22 != null)
		{
			sb.append("aniso_U22 = ");
			sb.append(aniso_U22);
			sb.append('\n');
		}
		if (aniso_U22_esd != null)
		{
			sb.append("aniso_U22_esd = ");
			sb.append(aniso_U22_esd);
			sb.append('\n');
		}
		if (aniso_U23 != null)
		{
			sb.append("aniso_U23 = ");
			sb.append(aniso_U23);
			sb.append('\n');
		}
		if (aniso_U23_esd != null)
		{
			sb.append("aniso_U23_esd = ");
			sb.append(aniso_U23_esd);
			sb.append('\n');
		}
		if (aniso_U33 != null)
		{
			sb.append("aniso_U33 = ");
			sb.append(aniso_U33);
			sb.append('\n');
		}
		if (aniso_U33_esd != null)
		{
			sb.append("aniso_U33_esd = ");
			sb.append(aniso_U33_esd);
			sb.append('\n');
		}
		if (aniso_ratio != null)
		{
			sb.append("aniso_ratio = ");
			sb.append(aniso_ratio);
			sb.append('\n');
		}
		if (attached_hydrogens != null)
		{
			sb.append("attached_hydrogens = ");
			sb.append(attached_hydrogens);
			sb.append('\n');
		}
		if (auth_asym_id != null)
		{
			sb.append("auth_asym_id = ");
			sb.append(auth_asym_id);
			sb.append('\n');
		}
		if (auth_atom_id != null)
		{
			sb.append("auth_atom_id = ");
			sb.append(auth_atom_id);
			sb.append('\n');
		}
		if (auth_comp_id != null)
		{
			sb.append("auth_comp_id = ");
			sb.append(auth_comp_id);
			sb.append('\n');
		}
		if (auth_seq_id != null)
		{
			sb.append("auth_seq_id = ");
			sb.append(auth_seq_id);
			sb.append('\n');
		}
		if (calc_attached_atom != null)
		{
			sb.append("calc_attached_atom = ");
			sb.append(calc_attached_atom);
			sb.append('\n');
		}
		if (calc_flag != null)
		{
			sb.append("calc_flag = ");
			sb.append(calc_flag);
			sb.append('\n');
		}
		if (chemical_conn_number != null)
		{
			sb.append("chemical_conn_number = ");
			sb.append(chemical_conn_number);
			sb.append('\n');
		}
		if (constraints != null)
		{
			sb.append("constraints = ");
			sb.append(constraints);
			sb.append('\n');
		}
		if (details != null)
		{
			sb.append("details = ");
			sb.append(details);
			sb.append('\n');
		}
		if (disorder_assembly != null)
		{
			sb.append("disorder_assembly = ");
			sb.append(disorder_assembly);
			sb.append('\n');
		}
		if (disorder_group != null)
		{
			sb.append("disorder_group = ");
			sb.append(disorder_group);
			sb.append('\n');
		}
		if (footnote_id != null)
		{
			sb.append("footnote_id = ");
			sb.append(footnote_id);
			sb.append('\n');
		}
		if (fract_x != null)
		{
			sb.append("fract_x = ");
			sb.append(fract_x);
			sb.append('\n');
		}
		if (fract_x_esd != null)
		{
			sb.append("fract_x_esd = ");
			sb.append(fract_x_esd);
			sb.append('\n');
		}
		if (fract_y != null)
		{
			sb.append("fract_y = ");
			sb.append(fract_y);
			sb.append('\n');
		}
		if (fract_y_esd != null)
		{
			sb.append("fract_y_esd = ");
			sb.append(fract_y_esd);
			sb.append('\n');
		}
		if (fract_z != null)
		{
			sb.append("fract_z = ");
			sb.append(fract_z);
			sb.append('\n');
		}
		if (fract_z_esd != null)
		{
			sb.append("fract_z_esd = ");
			sb.append(fract_z_esd);
			sb.append('\n');
		}
		if (group_PDB != null)
		{
			sb.append("group_PDB = ");
			sb.append(group_PDB);
			sb.append('\n');
		}
		if (id_ != null)
		{
			sb.append("id = ");
			sb.append(id_);
			sb.append('\n');
		}
		if (label_alt_id != null)
		{
			sb.append("label_alt_id = ");
			sb.append(label_alt_id);
			sb.append('\n');
		}
		if (label_asym_id != null)
		{
			sb.append("label_asym_id = ");
			sb.append(label_asym_id);
			sb.append('\n');
		}
		if (label_atom_id != null)
		{
			sb.append("label_atom_id = ");
			sb.append(label_atom_id);
			sb.append('\n');
		}
		if (label_comp_id != null)
		{
			sb.append("label_comp_id = ");
			sb.append(label_comp_id);
			sb.append('\n');
		}
		if (label_entity_id != null)
		{
			sb.append("label_entity_id = ");
			sb.append(label_entity_id);
			sb.append('\n');
		}
		if (label_seq_id != null)
		{
			sb.append("label_seq_id = ");
			sb.append(label_seq_id);
			sb.append('\n');
		}
		if (occupancy != null)
		{
			sb.append("occupancy = ");
			sb.append(occupancy);
			sb.append('\n');
		}
		if (occupancy_esd != null)
		{
			sb.append("occupancy_esd = ");
			sb.append(occupancy_esd);
			sb.append('\n');
		}
		if (pdbx_PDB_atom_name != null)
		{
			sb.append("pdbx_PDB_atom_name = ");
			sb.append(pdbx_PDB_atom_name);
			sb.append('\n');
		}
		if (pdbx_PDB_ins_code != null)
		{
			sb.append("pdbx_PDB_ins_code = ");
			sb.append(pdbx_PDB_ins_code);
			sb.append('\n');
		}
		if (pdbx_PDB_model_num != null)
		{
			sb.append("pdbx_PDB_model_num = ");
			sb.append(pdbx_PDB_model_num);
			sb.append('\n');
		}
		if (pdbx_PDB_residue_name != null)
		{
			sb.append("pdbx_PDB_residue_name = ");
			sb.append(pdbx_PDB_residue_name);
			sb.append('\n');
		}
		if (pdbx_PDB_residue_no != null)
		{
			sb.append("pdbx_PDB_residue_no = ");
			sb.append(pdbx_PDB_residue_no);
			sb.append('\n');
		}
		if (pdbx_PDB_strand_id != null)
		{
			sb.append("pdbx_PDB_strand_id = ");
			sb.append(pdbx_PDB_strand_id);
			sb.append('\n');
		}
		if (pdbx_auth_alt_id != null)
		{
			sb.append("pdbx_auth_alt_id = ");
			sb.append(pdbx_auth_alt_id);
			sb.append('\n');
		}
		if (pdbx_auth_atom_name != null)
		{
			sb.append("pdbx_auth_atom_name = ");
			sb.append(pdbx_auth_atom_name);
			sb.append('\n');
		}
		if (refinement_flags != null)
		{
			sb.append("refinement_flags = ");
			sb.append(refinement_flags);
			sb.append('\n');
		}
		if (refinement_flags_adp != null)
		{
			sb.append("refinement_flags_adp = ");
			sb.append(refinement_flags_adp);
			sb.append('\n');
		}
		if (refinement_flags_occupancy != null)
		{
			sb.append("refinement_flags_occupancy = ");
			sb.append(refinement_flags_occupancy);
			sb.append('\n');
		}
		if (refinement_flags_posn != null)
		{
			sb.append("refinement_flags_posn = ");
			sb.append(refinement_flags_posn);
			sb.append('\n');
		}
		if (restraints != null)
		{
			sb.append("restraints = ");
			sb.append(restraints);
			sb.append('\n');
		}
		if (symmetry_multiplicity != null)
		{
			sb.append("symmetry_multiplicity = ");
			sb.append(symmetry_multiplicity);
			sb.append('\n');
		}
		if (thermal_displace_type != null)
		{
			sb.append("thermal_displace_type = ");
			sb.append(thermal_displace_type);
			sb.append('\n');
		}
		if (type_symbol != null)
		{
			sb.append("type_symbol = ");
			sb.append(type_symbol);
			sb.append('\n');
		}
		return sb.toString();
	}

	public void clearValues() 
	{
		obj_id = null;
		B_equiv_geom_mean = null;
		B_equiv_geom_mean_esd = null;
		B_iso_or_equiv = null;
		B_iso_or_equiv_esd = null;
		Cartn_x = null;
		Cartn_x_esd = null;
		Cartn_y = null;
		Cartn_y_esd = null;
		Cartn_z = null;
		Cartn_z_esd = null;
		U_equiv_geom_mean = null;
		U_equiv_geom_mean_esd = null;
		U_iso_or_equiv = null;
		U_iso_or_equiv_esd = null;
		Wyckoff_symbol = null;
		adp_type = null;
		aniso_B11 = null;
		aniso_B11_esd = null;
		aniso_B12 = null;
		aniso_B12_esd = null;
		aniso_B13 = null;
		aniso_B13_esd = null;
		aniso_B22 = null;
		aniso_B22_esd = null;
		aniso_B23 = null;
		aniso_B23_esd = null;
		aniso_B33 = null;
		aniso_B33_esd = null;
		aniso_U11 = null;
		aniso_U11_esd = null;
		aniso_U12 = null;
		aniso_U12_esd = null;
		aniso_U13 = null;
		aniso_U13_esd = null;
		aniso_U22 = null;
		aniso_U22_esd = null;
		aniso_U23 = null;
		aniso_U23_esd = null;
		aniso_U33 = null;
		aniso_U33_esd = null;
		aniso_ratio = null;
		attached_hydrogens = null;
		auth_asym_id = null;
		auth_atom_id = null;
		auth_comp_id = null;
		auth_seq_id = null;
		calc_attached_atom = null;
		calc_flag = null;
		chemical_conn_number = null;
		constraints = null;
		details = null;
		disorder_assembly = null;
		disorder_group = null;
		footnote_id = null;
		fract_x = null;
		fract_x_esd = null;
		fract_y = null;
		fract_y_esd = null;
		fract_z = null;
		fract_z_esd = null;
		group_PDB = null;
		id_ = null;
		label_alt_id = null;
		label_asym_id = null;
		label_atom_id = null;
		label_comp_id = null;
		label_entity_id = null;
		label_seq_id = null;
		occupancy = null;
		occupancy_esd = null;
		pdbx_PDB_atom_name = null;
		pdbx_PDB_ins_code = null;
		pdbx_PDB_model_num = null;
		pdbx_PDB_residue_name = null;
		pdbx_PDB_residue_no = null;
		pdbx_PDB_strand_id = null;
		pdbx_auth_alt_id = null;
		pdbx_auth_atom_name = null;
		refinement_flags = null;
		refinement_flags_adp = null;
		refinement_flags_occupancy = null;
		refinement_flags_posn = null;
		restraints = null;
		symmetry_multiplicity = null;
		thermal_displace_type = null;
		type_symbol = null;
		
	}

	public boolean equals(Object other) 
	{
		if ( this.obj_id == null ) 
			return super.equals(other);
		return this.obj_id.longValue() == ((Atom_site)other).obj_id.longValue();
	}

	public int hashCode() 
	{
		return new HashCodeBuilder().append(getObj_id()).toHashCode();
	}

	public void toXml(java.io.OutputStreamWriter ow, String indent)
	{
		try
		{
			ow.write(indent + "<PDBx:atom_site>\n");
			if (B_equiv_geom_mean != null) ow.write(indent + " <PDBx:B_equiv_geom_mean>" + B_equiv_geom_mean + "</PDBx:B_equiv_geom_mean>\n");
			if (B_equiv_geom_mean_esd != null) ow.write(indent + " <PDBx:B_equiv_geom_mean_esd>" + B_equiv_geom_mean_esd + "</PDBx:B_equiv_geom_mean_esd>\n");
			if (B_iso_or_equiv != null) ow.write(indent + " <PDBx:B_iso_or_equiv>" + B_iso_or_equiv + "</PDBx:B_iso_or_equiv>\n");
			if (B_iso_or_equiv_esd != null) ow.write(indent + " <PDBx:B_iso_or_equiv_esd>" + B_iso_or_equiv_esd + "</PDBx:B_iso_or_equiv_esd>\n");
			if (Cartn_x != null) ow.write(indent + " <PDBx:Cartn_x>" + Cartn_x + "</PDBx:Cartn_x>\n");
			if (Cartn_x_esd != null) ow.write(indent + " <PDBx:Cartn_x_esd>" + Cartn_x_esd + "</PDBx:Cartn_x_esd>\n");
			if (Cartn_y != null) ow.write(indent + " <PDBx:Cartn_y>" + Cartn_y + "</PDBx:Cartn_y>\n");
			if (Cartn_y_esd != null) ow.write(indent + " <PDBx:Cartn_y_esd>" + Cartn_y_esd + "</PDBx:Cartn_y_esd>\n");
			if (Cartn_z != null) ow.write(indent + " <PDBx:Cartn_z>" + Cartn_z + "</PDBx:Cartn_z>\n");
			if (Cartn_z_esd != null) ow.write(indent + " <PDBx:Cartn_z_esd>" + Cartn_z_esd + "</PDBx:Cartn_z_esd>\n");
			if (U_equiv_geom_mean != null) ow.write(indent + " <PDBx:U_equiv_geom_mean>" + U_equiv_geom_mean + "</PDBx:U_equiv_geom_mean>\n");
			if (U_equiv_geom_mean_esd != null) ow.write(indent + " <PDBx:U_equiv_geom_mean_esd>" + U_equiv_geom_mean_esd + "</PDBx:U_equiv_geom_mean_esd>\n");
			if (U_iso_or_equiv != null) ow.write(indent + " <PDBx:U_iso_or_equiv>" + U_iso_or_equiv + "</PDBx:U_iso_or_equiv>\n");
			if (U_iso_or_equiv_esd != null) ow.write(indent + " <PDBx:U_iso_or_equiv_esd>" + U_iso_or_equiv_esd + "</PDBx:U_iso_or_equiv_esd>\n");
			if (Wyckoff_symbol != null) ow.write(indent + " <PDBx:Wyckoff_symbol>" + Wyckoff_symbol + "</PDBx:Wyckoff_symbol>\n");
			if (adp_type != null) ow.write(indent + " <PDBx:adp_type>" + adp_type + "</PDBx:adp_type>\n");
			if (aniso_B11 != null) ow.write(indent + " <PDBx:aniso_B11>" + aniso_B11 + "</PDBx:aniso_B11>\n");
			if (aniso_B11_esd != null) ow.write(indent + " <PDBx:aniso_B11_esd>" + aniso_B11_esd + "</PDBx:aniso_B11_esd>\n");
			if (aniso_B12 != null) ow.write(indent + " <PDBx:aniso_B12>" + aniso_B12 + "</PDBx:aniso_B12>\n");
			if (aniso_B12_esd != null) ow.write(indent + " <PDBx:aniso_B12_esd>" + aniso_B12_esd + "</PDBx:aniso_B12_esd>\n");
			if (aniso_B13 != null) ow.write(indent + " <PDBx:aniso_B13>" + aniso_B13 + "</PDBx:aniso_B13>\n");
			if (aniso_B13_esd != null) ow.write(indent + " <PDBx:aniso_B13_esd>" + aniso_B13_esd + "</PDBx:aniso_B13_esd>\n");
			if (aniso_B22 != null) ow.write(indent + " <PDBx:aniso_B22>" + aniso_B22 + "</PDBx:aniso_B22>\n");
			if (aniso_B22_esd != null) ow.write(indent + " <PDBx:aniso_B22_esd>" + aniso_B22_esd + "</PDBx:aniso_B22_esd>\n");
			if (aniso_B23 != null) ow.write(indent + " <PDBx:aniso_B23>" + aniso_B23 + "</PDBx:aniso_B23>\n");
			if (aniso_B23_esd != null) ow.write(indent + " <PDBx:aniso_B23_esd>" + aniso_B23_esd + "</PDBx:aniso_B23_esd>\n");
			if (aniso_B33 != null) ow.write(indent + " <PDBx:aniso_B33>" + aniso_B33 + "</PDBx:aniso_B33>\n");
			if (aniso_B33_esd != null) ow.write(indent + " <PDBx:aniso_B33_esd>" + aniso_B33_esd + "</PDBx:aniso_B33_esd>\n");
			if (aniso_U11 != null) ow.write(indent + " <PDBx:aniso_U11>" + aniso_U11 + "</PDBx:aniso_U11>\n");
			if (aniso_U11_esd != null) ow.write(indent + " <PDBx:aniso_U11_esd>" + aniso_U11_esd + "</PDBx:aniso_U11_esd>\n");
			if (aniso_U12 != null) ow.write(indent + " <PDBx:aniso_U12>" + aniso_U12 + "</PDBx:aniso_U12>\n");
			if (aniso_U12_esd != null) ow.write(indent + " <PDBx:aniso_U12_esd>" + aniso_U12_esd + "</PDBx:aniso_U12_esd>\n");
			if (aniso_U13 != null) ow.write(indent + " <PDBx:aniso_U13>" + aniso_U13 + "</PDBx:aniso_U13>\n");
			if (aniso_U13_esd != null) ow.write(indent + " <PDBx:aniso_U13_esd>" + aniso_U13_esd + "</PDBx:aniso_U13_esd>\n");
			if (aniso_U22 != null) ow.write(indent + " <PDBx:aniso_U22>" + aniso_U22 + "</PDBx:aniso_U22>\n");
			if (aniso_U22_esd != null) ow.write(indent + " <PDBx:aniso_U22_esd>" + aniso_U22_esd + "</PDBx:aniso_U22_esd>\n");
			if (aniso_U23 != null) ow.write(indent + " <PDBx:aniso_U23>" + aniso_U23 + "</PDBx:aniso_U23>\n");
			if (aniso_U23_esd != null) ow.write(indent + " <PDBx:aniso_U23_esd>" + aniso_U23_esd + "</PDBx:aniso_U23_esd>\n");
			if (aniso_U33 != null) ow.write(indent + " <PDBx:aniso_U33>" + aniso_U33 + "</PDBx:aniso_U33>\n");
			if (aniso_U33_esd != null) ow.write(indent + " <PDBx:aniso_U33_esd>" + aniso_U33_esd + "</PDBx:aniso_U33_esd>\n");
			if (aniso_ratio != null) ow.write(indent + " <PDBx:aniso_ratio>" + aniso_ratio + "</PDBx:aniso_ratio>\n");
			if (attached_hydrogens != null) ow.write(indent + " <PDBx:attached_hydrogens>" + attached_hydrogens + "</PDBx:attached_hydrogens>\n");
			if (auth_asym_id != null) ow.write(indent + " <PDBx:auth_asym_id>" + auth_asym_id + "</PDBx:auth_asym_id>\n");
			if (auth_atom_id != null) ow.write(indent + " <PDBx:auth_atom_id>" + auth_atom_id + "</PDBx:auth_atom_id>\n");
			if (auth_comp_id != null) ow.write(indent + " <PDBx:auth_comp_id>" + auth_comp_id + "</PDBx:auth_comp_id>\n");
			if (auth_seq_id != null) ow.write(indent + " <PDBx:auth_seq_id>" + auth_seq_id + "</PDBx:auth_seq_id>\n");
			if (calc_attached_atom != null) ow.write(indent + " <PDBx:calc_attached_atom>" + calc_attached_atom + "</PDBx:calc_attached_atom>\n");
			if (calc_flag != null) ow.write(indent + " <PDBx:calc_flag>" + calc_flag + "</PDBx:calc_flag>\n");
			if (chemical_conn_number != null) ow.write(indent + " <PDBx:chemical_conn_number>" + chemical_conn_number + "</PDBx:chemical_conn_number>\n");
			if (constraints != null) ow.write(indent + " <PDBx:constraints>" + constraints + "</PDBx:constraints>\n");
			if (details != null) ow.write(indent + " <PDBx:details>" + details + "</PDBx:details>\n");
			if (disorder_assembly != null) ow.write(indent + " <PDBx:disorder_assembly>" + disorder_assembly + "</PDBx:disorder_assembly>\n");
			if (disorder_group != null) ow.write(indent + " <PDBx:disorder_group>" + disorder_group + "</PDBx:disorder_group>\n");
			if (footnote_id != null) ow.write(indent + " <PDBx:footnote_id>" + footnote_id + "</PDBx:footnote_id>\n");
			if (fract_x != null) ow.write(indent + " <PDBx:fract_x>" + fract_x + "</PDBx:fract_x>\n");
			if (fract_x_esd != null) ow.write(indent + " <PDBx:fract_x_esd>" + fract_x_esd + "</PDBx:fract_x_esd>\n");
			if (fract_y != null) ow.write(indent + " <PDBx:fract_y>" + fract_y + "</PDBx:fract_y>\n");
			if (fract_y_esd != null) ow.write(indent + " <PDBx:fract_y_esd>" + fract_y_esd + "</PDBx:fract_y_esd>\n");
			if (fract_z != null) ow.write(indent + " <PDBx:fract_z>" + fract_z + "</PDBx:fract_z>\n");
			if (fract_z_esd != null) ow.write(indent + " <PDBx:fract_z_esd>" + fract_z_esd + "</PDBx:fract_z_esd>\n");
			if (group_PDB != null) ow.write(indent + " <PDBx:group_PDB>" + group_PDB + "</PDBx:group_PDB>\n");
			if (id_ != null) ow.write(indent + " <PDBx:id>" + id_ + "</PDBx:id>\n");
			if (label_alt_id != null) ow.write(indent + " <PDBx:label_alt_id>" + label_alt_id + "</PDBx:label_alt_id>\n");
			if (label_asym_id != null) ow.write(indent + " <PDBx:label_asym_id>" + label_asym_id + "</PDBx:label_asym_id>\n");
			if (label_atom_id != null) ow.write(indent + " <PDBx:label_atom_id>" + label_atom_id + "</PDBx:label_atom_id>\n");
			if (label_comp_id != null) ow.write(indent + " <PDBx:label_comp_id>" + label_comp_id + "</PDBx:label_comp_id>\n");
			if (label_entity_id != null) ow.write(indent + " <PDBx:label_entity_id>" + label_entity_id + "</PDBx:label_entity_id>\n");
			if (label_seq_id != null) ow.write(indent + " <PDBx:label_seq_id>" + label_seq_id + "</PDBx:label_seq_id>\n");
			if (occupancy != null) ow.write(indent + " <PDBx:occupancy>" + occupancy + "</PDBx:occupancy>\n");
			if (occupancy_esd != null) ow.write(indent + " <PDBx:occupancy_esd>" + occupancy_esd + "</PDBx:occupancy_esd>\n");
			if (pdbx_PDB_atom_name != null) ow.write(indent + " <PDBx:pdbx_PDB_atom_name>" + pdbx_PDB_atom_name + "</PDBx:pdbx_PDB_atom_name>\n");
			if (pdbx_PDB_ins_code != null) ow.write(indent + " <PDBx:pdbx_PDB_ins_code>" + pdbx_PDB_ins_code + "</PDBx:pdbx_PDB_ins_code>\n");
			if (pdbx_PDB_model_num != null) ow.write(indent + " <PDBx:pdbx_PDB_model_num>" + pdbx_PDB_model_num + "</PDBx:pdbx_PDB_model_num>\n");
			if (pdbx_PDB_residue_name != null) ow.write(indent + " <PDBx:pdbx_PDB_residue_name>" + pdbx_PDB_residue_name + "</PDBx:pdbx_PDB_residue_name>\n");
			if (pdbx_PDB_residue_no != null) ow.write(indent + " <PDBx:pdbx_PDB_residue_no>" + pdbx_PDB_residue_no + "</PDBx:pdbx_PDB_residue_no>\n");
			if (pdbx_PDB_strand_id != null) ow.write(indent + " <PDBx:pdbx_PDB_strand_id>" + pdbx_PDB_strand_id + "</PDBx:pdbx_PDB_strand_id>\n");
			if (pdbx_auth_alt_id != null) ow.write(indent + " <PDBx:pdbx_auth_alt_id>" + pdbx_auth_alt_id + "</PDBx:pdbx_auth_alt_id>\n");
			if (pdbx_auth_atom_name != null) ow.write(indent + " <PDBx:pdbx_auth_atom_name>" + pdbx_auth_atom_name + "</PDBx:pdbx_auth_atom_name>\n");
			if (refinement_flags != null) ow.write(indent + " <PDBx:refinement_flags>" + refinement_flags + "</PDBx:refinement_flags>\n");
			if (refinement_flags_adp != null) ow.write(indent + " <PDBx:refinement_flags_adp>" + refinement_flags_adp + "</PDBx:refinement_flags_adp>\n");
			if (refinement_flags_occupancy != null) ow.write(indent + " <PDBx:refinement_flags_occupancy>" + refinement_flags_occupancy + "</PDBx:refinement_flags_occupancy>\n");
			if (refinement_flags_posn != null) ow.write(indent + " <PDBx:refinement_flags_posn>" + refinement_flags_posn + "</PDBx:refinement_flags_posn>\n");
			if (restraints != null) ow.write(indent + " <PDBx:restraints>" + restraints + "</PDBx:restraints>\n");
			if (symmetry_multiplicity != null) ow.write(indent + " <PDBx:symmetry_multiplicity>" + symmetry_multiplicity + "</PDBx:symmetry_multiplicity>\n");
			if (thermal_displace_type != null) ow.write(indent + " <PDBx:thermal_displace_type>" + thermal_displace_type + "</PDBx:thermal_displace_type>\n");
			if (type_symbol != null) ow.write(indent + " <PDBx:type_symbol>" + type_symbol + "</PDBx:type_symbol>\n");
			ow.write(indent + "</PDBx:atom_site>\n");
		}
		catch (java.io.IOException e)
		{
			e.printStackTrace();
		}
	}
}
