package org.pdb.ormapping.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Stack;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

public class SchemaReader
{

   public static Node getFirstNonTextNode(Node aTerm)
   {
      NodeList nl = aTerm.getChildNodes();
      Node ans = null;
      int nll = nl.getLength();
      for (int j = 0; j < nll; j++)
      {
         Node foo = nl.item(j);
         if (foo.getNodeType() != Node.TEXT_NODE)
         {
            return foo;
         }
      }
      return ans;
   }

   protected static Stack complexTypesStack = new Stack();

   protected static Map currentComplexTypes = new HashMap();

   protected static Node findComplexType(String aComplexTypeNodeName)
   {
      String actnn = aComplexTypeNodeName;
      if (actnn.startsWith("PDBx:"))
      {
         actnn = actnn.substring(5);
      }
      Node ans = null;
      Map aMap = currentComplexTypes;
      if (aMap.containsKey(actnn))
      {
         ans = (Node) aMap.get(actnn);
      }
      else
      {
         Stack ctStackCopy = (Stack) complexTypesStack.clone();
         while (!ctStackCopy.empty())
         {
            aMap = (HashMap) ctStackCopy.pop();
            if (aMap.containsKey(actnn))
            {
               ans = (Node) aMap.get(actnn);
               break;
            }
         }
      }
      return ans;
   }

   protected static String getDoc(Node aNode)
   {
      String ans = "";
      StringBuffer sb = new StringBuffer();
      NodeList kids = aNode.getChildNodes();
      int klen = kids.getLength();
      for (int j = 0; j < klen; j++)
      {
         Node aKid = kids.item(j);
         if (aKid.getNodeType() == Node.ELEMENT_NODE && aKid.getNodeName().startsWith("xsd:documentation"))
         {
            NodeList kidkids = aKid.getChildNodes();
            int kklen = kidkids.getLength();
            for (int k = 0; k < kklen; k++)
            {
               Node aKidKid = kidkids.item(k);
               if (aKidKid.getNodeType() == Node.TEXT_NODE)
               {
                  sb.append(aKidKid.getNodeValue());
               }
               // showNodeInfo(aKidKid, "getDoc");
            }
            ans = sb.toString().trim();
         }
      }
      return ans;
   }

   protected static String getNodeType(Node aNode)
   {
      return getNodeAttrName(aNode, "type");
   }

   protected static String getNodeName(Node aNode)
   {
      return getNodeAttrName(aNode, "name");
   }

   protected static String getNodeBase(Node aNode)
   {
      return getNodeAttrName(aNode, "base");
   }

   protected static String getNodeAttrName(Node aNode, String attrName)
   {
      String ans = "";
      NamedNodeMap attrs = aNode.getAttributes();
      if (attrs != null)
      {
         Node typeNode = attrs.getNamedItem(attrName);
         if (typeNode != null)
         {
            ans = typeNode.getNodeValue();
         }
      }
      return ans;
   }

   protected static void showNodeInfo(Node aNode, String msg)
   {
      // System.err.println("In " + msg + " - " + aNode.getNodeName() + " " +
      // getNodeName(aNode) + " " + getNodeType(aNode));
   }

   protected static List allEnums = new ArrayList();

   protected static CifEnum processEnum(Node elementNode, Node enumNode, Table currentTable)
   {
      showNodeInfo(enumNode, "processEnum");
      CifEnum e = new CifEnum(currentTable.getName(), getNodeName(elementNode), getNodeBase(enumNode));
      NodeList kids = enumNode.getChildNodes();
      int klen = kids.getLength();
      boolean saveIt = false;
      for (int j = 0; j < klen; j++)
      {
         Node aKid = kids.item(j);
         if (aKid.getNodeType() == Node.ELEMENT_NODE && aKid.getNodeName().equals("xsd:enumeration"))
         {
            String aval = getNodeAttrName(aKid, "value");
            e.addValue(aval);
            // System.err.println("added " + aval);
            saveIt = true;
         }
      }
      if (saveIt)
      {
         allEnums.add(e);
         return e;
      }
      else
      {
         return null;
      }
   }

   protected static void processUnion(Node elementNode, Node simpleNode, Table currentTable)
   {
      showNodeInfo(simpleNode, "processUnion");
      NodeList kids = simpleNode.getChildNodes();
      int klen = kids.getLength();
      for (int j = 0; j < klen; j++)
      {
         Node aKid = kids.item(j);
         if (aKid.getNodeType() == Node.ELEMENT_NODE && aKid.getNodeName().equals("xsd:simpleType"))
         {
            processSimpleType(elementNode, aKid, currentTable);
            return; // only worry about the first option in a union
         }
         else if (aKid.getNodeType() == Node.ELEMENT_NODE)
         {
            System.err.println("Error - processUnion didn't handle: " + aKid.getNodeName());
         }
      }
   }

   protected static void processSimpleType(Node elementNode, Node simpleNode, Table currentTable)
   {
      showNodeInfo(simpleNode, "processSimpleType");
      NodeList kids = simpleNode.getChildNodes();
      int klen = kids.getLength();
      Field f = null;
      for (int j = 0; j < klen; j++)
      {
         Node aKid = kids.item(j);
         if (aKid.getNodeType() == Node.ELEMENT_NODE && aKid.getNodeName().equals("xsd:restriction"))
         {
            String base = getNodeBase(aKid);
            f = addField(elementNode, currentTable, base);
            f.setEnumValues(processEnum(elementNode, aKid, currentTable));
         }
         else if (aKid.getNodeType() == Node.ELEMENT_NODE && aKid.getNodeName().equals("xsd:union"))
         {
            processUnion(elementNode, aKid, currentTable);
         }
         else if (aKid.getNodeType() == Node.ELEMENT_NODE)
         {
            System.err.println("Error - processSimpleType didn't handle: " + aKid.getNodeName());
         }
      }
   }

   protected static void processElement(Node aNode, Table currentTable)
   {
      showNodeInfo(aNode, "processElement");
      String nodeType = getNodeType(aNode);
      if (nodeType.startsWith("PDBx:"))
      {
         Node aCt = findComplexType(nodeType);
         if (aCt != null)
         {
            processComplexType(aNode, aCt, currentTable);
         }
         else
         {
            System.err.println("Error: couldn't find complex type for element: " + nodeType);
         }
         return;
      }
      else if (nodeType.startsWith("xsd:")) // simple element
      {
         addField(aNode, currentTable);
         return;
      }
      NodeList kids = aNode.getChildNodes();
      int klen = kids.getLength();

      // check for xsd:complexType anonymous
      if (klen == 1)
      {
         Node aKid = kids.item(0);
         if (aKid.getNodeType() == Node.ELEMENT_NODE && aKid.getNodeName().equals("xsd:complexType"))
         {
            processComplexType(aNode, aKid, currentTable);
            return;
         }
      }

      // check for doc
      String doc = null;
      for (int j = 0; j < klen; j++)
      {
         Node aKid = kids.item(j);
         if (aKid.getNodeType() == Node.ELEMENT_NODE && aKid.getNodeName().startsWith("xsd:annotation"))
         {
            doc = getDoc(aKid);
         }
      }

      for (int j = 0; j < klen; j++)
      {
         Node aKid = kids.item(j);
         if (aKid.getNodeType() == Node.ELEMENT_NODE && aKid.getNodeName().equals("xsd:element"))
         {
            processElement(aKid, currentTable);
         }
         else if (aKid.getNodeType() == Node.ELEMENT_NODE && aKid.getNodeName().equals("xsd:complexType"))
         {
            Node aNameNode = aKid.getAttributes().getNamedItem("name");
            if (aNameNode != null)
            {
               currentComplexTypes.put(aNameNode.getNodeValue(), aKid);
            }
            else
            {
               processComplexType(aNode, aKid, currentTable);
            }
         }
         else if (aKid.getNodeType() == Node.ELEMENT_NODE && aKid.getNodeName().equals("xsd:sequence"))
         {
            processSequence(aNode, getFirstNonTextNode(aKid), currentTable, doc);
         }
         else if (aKid.getNodeType() == Node.ELEMENT_NODE && aKid.getNodeName().startsWith("xsd:attribute"))
         {
            addAttribute(aKid, currentTable);
         }
         else if (aKid.getNodeType() == Node.ELEMENT_NODE && aKid.getNodeName().startsWith("xsd:simpleType"))
         {
            processSimpleType(aNode, aKid, currentTable);
         }
         else if (aKid.getNodeType() == Node.ELEMENT_NODE && aKid.getNodeName().startsWith("xsd:annotation"))
         {
            // ignore these - we did them up front
         }
         else if (aKid.getNodeType() == Node.ELEMENT_NODE)
         {
            System.err.println("Error - processElement didn't handle: " + aKid.getNodeName());
         }
      }
   }

   protected static void addAttribute(Node aNode, Table currentTable)
   {
      showNodeInfo(aNode, "addAttribute");
      // check for doc
      String doc = null;
      NodeList kids = aNode.getChildNodes();
      int klen = kids.getLength();
      for (int j = 0; j < klen; j++)
      {
         Node aKid = kids.item(j);
         if (aKid.getNodeType() == Node.ELEMENT_NODE && aKid.getNodeName().startsWith("xsd:annotation"))
         {
            doc = getDoc(aKid);
         }
      }
      // create and set up field
      Field f = new Field(currentTable, true);
      NamedNodeMap attrs = aNode.getAttributes();
      Node nameNode = attrs.getNamedItem("name");
      if (nameNode != null)
      {
         f.setName(nameNode.getNodeValue());
      }
      else
      {
         System.err.println("Serious problem - no name for attribute in table: " + currentTable.getName());
      }
      Node typeNode = attrs.getNamedItem("type");
      if (typeNode != null)
      {
         f.setDataType(typeNode.getNodeValue());
      }
      else
      {
         processElement(aNode, currentTable);
         return;
      }
      if (doc != null)
      {
         f.setDescription(doc);
      }
      f.setDataType(getMappedType(f.getDataType()));
      currentTable.addField(f);
   }

   protected static void addField(Node aNode, Table currentTable)
   {
      showNodeInfo(aNode, "addField");
      // check for doc
      String doc = null;
      NodeList kids = aNode.getChildNodes();
      int klen = kids.getLength();
      for (int j = 0; j < klen; j++)
      {
         Node aKid = kids.item(j);
         if (aKid.getNodeType() == Node.ELEMENT_NODE && aKid.getNodeName().startsWith("xsd:annotation"))
         {
            doc = getDoc(aKid);
         }
      }
      // create and set up field
      Field f = new Field(currentTable, false);
      NamedNodeMap attrs = aNode.getAttributes();
      Node nameNode = attrs.getNamedItem("name");
      if (nameNode != null)
      {
         f.setName(nameNode.getNodeValue());
      }
      else
      {
         System.err.println("Serious problem - no name for field in table: " + currentTable.getName());
      }
      Node typeNode = attrs.getNamedItem("type");
      if (typeNode != null)
      {
         f.setDataType(typeNode.getNodeValue());
      }
      else
      {
         if (f.getName() == null)
         {
            System.err.println("Serious problem - no type for field in table: " + currentTable.getName());
         }
         else
         {
            System.err.println("Serious problem - no type for field " + f.getName() + " in table: "
                  + currentTable.getName());
         }
      }
      if (doc != null)
      {
         f.setDescription(doc);
      }
      f.setDataType(getMappedType(f.getDataType()));

      Node nillableNode = attrs.getNamedItem("nillable");
      if (nillableNode != null && nillableNode.getNodeValue().equalsIgnoreCase("true"))
      {
         f.setIsNullable(true);
      }

      currentTable.addField(f);
   }

   protected static Field addField(Node aNode, Table currentTable, String nodeType)
   {
      showNodeInfo(aNode, "addField");
      // check for doc
      if (nodeType == null)
      {
         System.err.println("null nodeType! " + currentTable.getName());
      }
      String doc = null;
      NodeList kids = aNode.getChildNodes();
      int klen = kids.getLength();
      for (int j = 0; j < klen; j++)
      {
         Node aKid = kids.item(j);
         if (aKid.getNodeType() == Node.ELEMENT_NODE && aKid.getNodeName().startsWith("xsd:annotation"))
         {
            doc = getDoc(aKid);
         }
      }
      // create and set up field
      Field f = new Field(currentTable, false);
      f.setDataType(nodeType);
      NamedNodeMap attrs = aNode.getAttributes();
      Node nameNode = attrs.getNamedItem("name");
      if (nameNode != null)
      {
         f.setName(nameNode.getNodeValue());
      }
      else
      {
         System.err.println("Serious problem - no name for field in table: " + currentTable.getName());
      }
      if (doc != null)
      {
         f.setDescription(doc);
      }
      f.setDataType(getMappedType(f.getDataType()));
      String aMin = getMinValue(aNode);
      if (aMin != null)
      {
         f.setMinValue(aMin);
      }
      String aMax = getMaxValue(aNode);
      if (aMax != null)
      {
         f.setMaxValue(aMax);
      }
      currentTable.addField(f);
      return f;
   }

   protected static void processSimpleContent(Node elementNode, Node aNode, Table currentTable)
   {
      showNodeInfo(aNode, "processSimpleContent");
      NodeList kids = aNode.getChildNodes();
      int klen = kids.getLength();
      for (int j = 0; j < klen; j++)
      {
         Node aKid = kids.item(j);
         if (aKid.getNodeType() == Node.ELEMENT_NODE && aKid.getNodeName().equals("xsd:extension"))
         {
            String base = getNodeBase(aKid);
            Field f = addField(elementNode, currentTable, base);
            String units = getUnitsOfMeasure(aKid);
            if (units != null)
            {
               f.setUnits(units);
            }
         }
         else if (aKid.getNodeType() == Node.ELEMENT_NODE)
         {
            System.err.println("Error - processSimpleContent didn't handle: " + aKid.getNodeName());
         }
      }
   }

   protected static String getUnitsOfMeasure(Node aNode)
   {
      String ans = null;
      showNodeInfo(aNode, "getUnitsOfMeasure");
      NodeList kids = aNode.getChildNodes();
      int klen = kids.getLength();
      for (int j = 0; j < klen; j++)
      {
         Node aKid = kids.item(j);
         if (aKid.getNodeType() == Node.ELEMENT_NODE && aKid.getNodeName().equals("xsd:attribute"))
         {
            NamedNodeMap attrs = aKid.getAttributes();
            Node nameNode = attrs.getNamedItem("name");
            if (nameNode != null && nameNode.getNodeValue().equalsIgnoreCase("units"))
            {
               Node fixedNode = attrs.getNamedItem("fixed");
               if (fixedNode != null)
               {
                  ans = fixedNode.getNodeValue();
               }
            }
         }
         else if (aKid.getNodeType() == Node.ELEMENT_NODE)
         {
            System.err.println("Error - getUnitsOfMeasure didn't handle: " + aKid.getNodeName());
         }
      }

      return ans;
   }

   protected static String getMinValue(Node aNode)
   {
      String ans = null;
      float curlow = Float.MAX_VALUE;
      if (aNode.getNodeType() == Node.ELEMENT_NODE)
      {
         Element e = (Element) aNode;
         NodeList nl = e.getElementsByTagName("xsd:minInclusive");
         int klen = nl.getLength();
         for (int j = 0; j < klen; j++)
         {
            Node aKid = nl.item(j);
            NamedNodeMap attrs = aKid.getAttributes();
            Node valNode = attrs.getNamedItem("value");
            if (valNode != null)
            {
               try
               {
                  float alow = Float.parseFloat(valNode.getNodeValue());
                  if (alow < curlow)
                  {
                     ans = valNode.getNodeValue();
                     curlow = alow;
                  }
               }
               catch (Exception ex1)
               {
                  ex1.printStackTrace();
               }
            }
         }
         nl = e.getElementsByTagName("xsd:minExclusive");
         klen = nl.getLength();
         for (int j = 0; j < klen; j++)
         {
            Node aKid = nl.item(j);
            NamedNodeMap attrs = aKid.getAttributes();
            Node valNode = attrs.getNamedItem("value");
            if (valNode != null)
            {
               try
               {
                  float alow = Float.parseFloat(valNode.getNodeValue());
                  if (alow < curlow)
                  {
                     ans = valNode.getNodeValue();
                     curlow = alow;
                  }
               }
               catch (Exception ex1)
               {
                  ex1.printStackTrace();
               }
            }
         }
      }
      return ans;
   }

   protected static String getMaxValue(Node aNode)
   {
      String ans = null;
      float curhigh = Float.MIN_VALUE;
      if (aNode.getNodeType() == Node.ELEMENT_NODE)
      {
         Element e = (Element) aNode;
         NodeList nl = e.getElementsByTagName("xsd:maxInclusive");
         int klen = nl.getLength();
         for (int j = 0; j < klen; j++)
         {
            Node aKid = nl.item(j);
            NamedNodeMap attrs = aKid.getAttributes();
            Node valNode = attrs.getNamedItem("value");
            if (valNode != null)
            {
               try
               {
                  float ahigh = Float.parseFloat(valNode.getNodeValue());
                  if (ahigh > curhigh)
                  {
                     ans = valNode.getNodeValue();
                     curhigh = ahigh;
                  }
               }
               catch (Exception ex1)
               {
                  ex1.printStackTrace();
               }
            }
         }
         nl = e.getElementsByTagName("xsd:maxExclusive");
         klen = nl.getLength();
         for (int j = 0; j < klen; j++)
         {
            Node aKid = nl.item(j);
            NamedNodeMap attrs = aKid.getAttributes();
            Node valNode = attrs.getNamedItem("value");
            if (valNode != null)
            {
               try
               {
                  float ahigh = Float.parseFloat(valNode.getNodeValue());
                  if (ahigh > curhigh)
                  {
                     ans = valNode.getNodeValue();
                     curhigh = ahigh;
                  }
               }
               catch (Exception ex1)
               {
                  ex1.printStackTrace();
               }
            }
         }
      }
      return ans;
   }

   protected static void processComplexType(Node elementNode, Node aNode, Table currentTable)
   {
      showNodeInfo(aNode, "processComplexType");
      NodeList kids = aNode.getChildNodes();
      int klen = kids.getLength();
      // check for doc
      String doc = null;
      for (int j = 0; j < klen; j++)
      {
         Node aKid = kids.item(j);
         if (aKid.getNodeType() == Node.ELEMENT_NODE && aKid.getNodeName().startsWith("xsd:annotation"))
         {
            doc = getDoc(aKid);
         }
      }

      // check for sequence, elements
      for (int j = 0; j < klen; j++)
      {
         Node aKid = kids.item(j);
         if (aKid.getNodeType() == Node.ELEMENT_NODE && aKid.getNodeName().equals("xsd:all"))
         {
            processElementOrComplexType(elementNode, aKid, currentTable);
         }
         else if (aKid.getNodeType() == Node.ELEMENT_NODE && aKid.getNodeName().startsWith("xsd:sequence"))
         {
            processSequence(elementNode, getFirstNonTextNode(aKid), currentTable, doc);
         }
         else if (aKid.getNodeType() == Node.ELEMENT_NODE && aKid.getNodeName().startsWith("xsd:element"))
         {
            processElement(aKid, currentTable);
         }
         else if (aKid.getNodeType() == Node.ELEMENT_NODE && aKid.getNodeName().startsWith("xsd:annotation"))
         {
            // ignore these - we did them up front
         }
         else if (aKid.getNodeType() == Node.ELEMENT_NODE && aKid.getNodeName().startsWith("xsd:attribute"))
         {
            addAttribute(aKid, currentTable);
         }
         else if (aKid.getNodeType() == Node.ELEMENT_NODE && aKid.getNodeName().startsWith("xsd:simpleContent"))
         {
            processSimpleContent(elementNode, aKid, currentTable);
         }
         else if (aKid.getNodeType() == Node.ELEMENT_NODE)
         {
            System.err.println("Error - processComplexType didn't handle: " + aKid.getNodeName());
         }
      }

   }

   protected static void processSequence(Node elementNode, Node aNode, Table parentTable, String doc)
   {
      showNodeInfo(aNode, "processSequence");
      Table newTable = new Table(parentTable);
      if (doc != null && doc.trim().length() > 0)
      {
         newTable.setDescription(doc);
      }
      newTable.setName(getNodeName(aNode));
      // newTable.setKeyField((String)singleFieldKeys.get(newTable.getName()));
      newTable.setKeyField(null); // for now generate obj_ids for every table
      parentTable.addTable(newTable);
      processElementOrComplexType(elementNode, aNode, newTable);
   }

   protected static Map typeMap = null;

   protected static void initTypeMap()
   {
      typeMap = new HashMap();
      typeMap.put("xsd:date", "java.util.Date");
      typeMap.put("xsd:string", "java.lang.String");
      typeMap.put("xsd:decimal", "java.lang.Float");
      typeMap.put("xsd:integer", "java.lang.Integer");
   }

   protected static String getMappedType(String t)
   {
      String ans = t;
      if (typeMap.containsKey(t))
      {
         ans = (String) typeMap.get(t);
      }
      return ans;
   }

   protected static void processElementOrComplexType(Node elementNode, Node aNode, Table currentTable)
   {
      if (typeMap == null) initTypeMap();
      showNodeInfo(aNode, "processElementOrComplexType");
      if (aNode.getNodeType() == Node.ELEMENT_NODE && aNode.getNodeName().equals("xsd:element"))
      {
         processElement(aNode, currentTable);
      }
      else if ((aNode.getNodeType() == Node.ELEMENT_NODE && aNode.getNodeName().equals("xsd:complexType"))
            || (aNode.getNodeType() == Node.ELEMENT_NODE && aNode.getNodeName().equals("xsd:all")))
      {
         processComplexType(elementNode, aNode, currentTable);
      }
   }

   protected static Map singleFieldKeys = new HashMap();

   protected static Map getSingleFieldKeys(Document d)
   {
      singleFieldKeys.clear();
      NodeList nl = d.getElementsByTagName("xsd:key");
      int klen = nl.getLength();
      String aField = null;
      String aTable = null;
      for (int j = 0; j < klen; j++)
      {
         Node aKeyNode = nl.item(j);
         NodeList flds = ((Element) aKeyNode).getElementsByTagName("xsd:field");
         if (flds.getLength() != 1) continue;
         aField = getNodeAttrName(flds.item(0), "xpath");
         if (aField.startsWith("@"))
         {
            aField = aField.substring(1);
         }
         flds = ((Element) aKeyNode).getElementsByTagName("xsd:selector");
         if (flds.getLength() != 1) continue;
         aTable = getNodeAttrName(flds.item(0), "xpath");
         int pos = aTable.lastIndexOf('/');
         if (pos > 0)
         {
            aTable = aTable.substring(pos + 1);
         }
         // System.err.println(aTable + ", " + aField);
         singleFieldKeys.put(aTable, aField);
      }
      return singleFieldKeys;
   }

   // public static String baseDir = "C:\\Documents and Settings\\Owner\\My
   // Documents\\devel\\jaxbPdb";
   // public static String baseDir = "C:\\jaxbPdb";
   public static String defaultSchemaName = "pdbx-v1.000.xsd";

   public static File getSchemaDir(String schemaName)
   {
      File ans = null;
      String sn = schemaName;
      if (sn == null)
      {
         sn = defaultSchemaName;
      }
      File schemaFile = new File(sn);
      if (!schemaFile.exists())
      {
         throw new RuntimeException("XML Schema file does NOT exist: " + sn);
      }
      ans = schemaFile.getParentFile();
      if (ans == null)
      {
         try
         {
            String cn = schemaFile.getCanonicalPath();
            int i = cn.lastIndexOf(File.separatorChar);
            if (i >= 0)
            {
               System.err.println("trying to get file with: " + cn.substring(0,i));
               ans = new File(cn.substring(0,i));
               System.err.println("got file: " + ans.getCanonicalPath());
            }
         }
         catch (IOException e)
         {
            // TODO Auto-generated catch block
            e.printStackTrace();
         }
      }
      return ans;
   }

   public static Table getTopTable(String schemaName)
   {
      Table ans = null;
      String sn = schemaName;
      if (sn == null)
      {
         sn = defaultSchemaName;
      }
      try
      {
         File schemaFile = new File(sn);
         if (!schemaFile.exists())
         {
            throw new RuntimeException("XML Schema file does NOT exist: " + sn);
         }
         DocumentBuilderFactory builderFactory = DocumentBuilderFactory.newInstance();
         DocumentBuilder builder = builderFactory.newDocumentBuilder();
         InputStream in = new FileInputStream(sn);
         Document aDoc = builder.parse(in);
         in.close();
         in = null;
         aDoc.normalize();
         getSingleFieldKeys(aDoc);

         complexTypesStack.push(currentComplexTypes);
         Node theSchema = aDoc.getDocumentElement();
         Node topElement = null;
         NodeList kids = theSchema.getChildNodes();
         int klen = kids.getLength();
         for (int j = 0; j < klen; j++)
         {
            Node aKid = kids.item(j);
            if (aKid.getNodeType() == Node.ELEMENT_NODE && aKid.getNodeName().equals("xsd:element"))
            {
               if (topElement != null)
               {
                  System.err.println("Error - multiple top level elements!");
                  System.exit(1);
               }
               else
               {
                  topElement = aKid;
               }
            }
            else if (aKid.getNodeType() == Node.ELEMENT_NODE && aKid.getNodeName().equals("xsd:complexType"))
            {
               currentComplexTypes.put(aKid.getAttributes().getNamedItem("name").getNodeValue(), aKid);
            }
         }
         Table topTable = new Table(null);
         topTable.setDescription("Top level of PDB Structure");
         topTable.setName(topElement.getAttributes().getNamedItem("name").getNodeValue());

         // now process an element
         processElementOrComplexType(null, topElement, topTable);

         // System.err.println("top table is:\n\n" + topTable.toString());
         ans = topTable;
      }
      catch (MalformedURLException e)
      {
         e.printStackTrace();
      }
      catch (IOException e)
      {
         e.printStackTrace();
      }
      catch (ParserConfigurationException e)
      {
         e.printStackTrace();
      }
      catch (SAXException e)
      {
         e.printStackTrace();
      }
      return ans;
   }

   public static void generateBeans(String schemaName)
   {
      String sn = schemaName;
      if (sn == null)
      {
         sn = defaultSchemaName;
      }
      BeanBuilder.generateJavaBeans(getTopTable(sn), getSchemaDir(sn));
   }

   public static void generateSchema(String workingDirIn, String mappingName, String propsName, String outputName)
   {
      String myWorkingDirIn = workingDirIn;
      if (myWorkingDirIn == null)
      {
         myWorkingDirIn = ".";
      }
      String myMappingName = mappingName;
      if (myMappingName == null)
      {
         myMappingName = "hibernate-mapping.xml";
      }
      String myPropsName = propsName;
      if (myPropsName == null)
      {
         myPropsName = "hibernate.properties";
      }
      String myOutputName = outputName;
      if (myOutputName == null)
      {
         myOutputName = "pdb_cif_schema.sql";
      }
      File workingDir = new File(myWorkingDirIn);
      if (!workingDir.exists() || !workingDir.isDirectory())
      {
         throw new RuntimeException("Working directory does NOT exist or is NOT a directory: " + myWorkingDirIn);
      }
      File mappingFile = new File(myWorkingDirIn + java.io.File.separator + myMappingName);
      if (!mappingFile.exists())
      {
         throw new RuntimeException("Hibernate Mapping file does NOT exist: " + myMappingName);
      }
      File propsFile = new File(myWorkingDirIn + java.io.File.separator + myPropsName);
      if (!propsFile.exists())
      {
         throw new RuntimeException("Hibernate Properties file does NOT exist: " + myPropsName);
      }

      String[] schemaExportArgs = new String[6];
      schemaExportArgs[0] = "--text";
      schemaExportArgs[1] = "--output=" + myWorkingDirIn + java.io.File.separator + myOutputName;
      schemaExportArgs[2] = "--format";
      schemaExportArgs[3] = "--delimiter=;";
      schemaExportArgs[4] = "--properties=" + myWorkingDirIn + java.io.File.separator + myPropsName;
      schemaExportArgs[5] = myWorkingDirIn + java.io.File.separator + myMappingName;
      org.hibernate.tool.hbm2ddl.SchemaExport.main(schemaExportArgs);
   }

   public static void main(String[] args)
   {
      if (args.length > 0 && args[0].equalsIgnoreCase("-generateSchema"))
      {
         if (args.length > 2)
         {
            generateSchema(args[1], args[2], args[3], args[4]);
         }
         else
         {
            System.err
                  .println("Usage - file argument not provided - use \"-generateSchema\" <working directory> <Mapping file> <Properties file> <Output file>");
         }
      }
      else
      {
         if (args.length > 0)
         {
            generateBeans(args[0]);
         }
         else
         {
            System.err.println("Usage - argument not provided - use \"-generateSchema\" or full path to XML Schema.");
         }
      }
   }

}